import tkinter as tk
from tkinter import ttk, messagebox
from . import model
from modules.einstellungen import model as settings_model
import datetime

def parse_datum(datum_raw):
    """Akzeptiert viele Formate, gibt ISO-String zurück oder wirft Exception."""
    if not datum_raw.strip():
        return datetime.date.today().isoformat()
    for fmt in ("%Y-%m-%d", "%d.%m.%Y", "%Y.%m.%d", "%d-%m-%Y", "%d/%m/%Y"):
        try:
            d = datetime.datetime.strptime(datum_raw.strip(), fmt)
            return d.date().isoformat()
        except Exception:
            continue
    raise ValueError

def open():
    model.init_db()
    win = tk.Toplevel()
    win.title("Mengen- & Chargenverwaltung")
    win.geometry("950x650")

    # Chargen-Liste
    tk.Label(win, text="Aktuelle Chargen:", font=("Segoe UI", 12, "bold")).pack(pady=5)
    columns = ("ID", "Bezeichnung", "Start", "Menge gesamt", "Menge rest")
    tree = ttk.Treeview(win, columns=columns, show="headings", height=8)
    for col in columns:
        tree.heading(col, text=col)
        tree.column(col, width=140)
    tree.pack(pady=5, padx=12, fill="x")

    def refresh_chargen():
        tree.delete(*tree.get_children())
        for row in model.get_chargen():
            tree.insert("", "end", values=row)
    refresh_chargen()

    # Neue Charge anlegen
    def open_add_charge():
        dlg = tk.Toplevel(win)
        dlg.title("Neue Charge")
        dlg.geometry("410x320")
        fields = [
            ("Bezeichnung", "bezeichnung"),
            ("Start-Datum", "start_datum"),
            ("End-Datum", "end_datum"),
            ("Gesamtmenge (g)", "menge_gesamt"),
            ("Kommentar", "kommentar"),
        ]
        entries = {}
        for idx, (label, key) in enumerate(fields):
            tk.Label(dlg, text=label+":", anchor="w").grid(row=idx, column=0, sticky="w", padx=8, pady=4)
            entry = tk.Entry(dlg, width=26)
            entry.grid(row=idx, column=1, pady=4)
            entries[key] = entry

        def save():
            data = {k: v.get().strip() for k, v in entries.items()}
            try:
                data["menge_gesamt"] = float(data["menge_gesamt"])
            except Exception:
                messagebox.showerror("Fehler", "Gesamtmenge muss eine Zahl sein.")
                return
            model.add_charge(data)
            messagebox.showinfo("Erfolg", "Charge angelegt.")
            dlg.destroy()
            refresh_chargen()

        tk.Button(dlg, text="Speichern", command=save).grid(row=len(fields), column=0, pady=20)
        tk.Button(dlg, text="Abbrechen", command=dlg.destroy).grid(row=len(fields), column=1, pady=20)

    tk.Button(win, text="Neue Charge", command=open_add_charge).pack(pady=4)

    # Abgaben-Liste
    tk.Label(win, text="Abgaben (letzte 20):", font=("Segoe UI", 12, "bold")).pack(pady=7)
    abgabe_cols = ("ID", "Mitglied-ID", "Charge-ID", "Datum", "Menge (g)", "Bearbeiter")
    tree_a = ttk.Treeview(win, columns=abgabe_cols, show="headings", height=10)
    for col in abgabe_cols:
        tree_a.heading(col, text=col)
        tree_a.column(col, width=120)
    tree_a.pack(pady=5, padx=12, fill="x")

    def refresh_abgaben():
        tree_a.delete(*tree_a.get_children())
        for row in model.get_abgaben()[:20]:
            tree_a.insert("", "end", values=row)
    refresh_abgaben()

    # Neue Abgabe
    def open_add_abgabe():
        dlg = tk.Toplevel(win)
        dlg.title("Neue Abgabe")
        dlg.geometry("430x370")
        fields = [
            ("Mitglied-ID", "mitglied_id"),
            ("Charge-ID", "charge_id"),
            ("Datum (JJJJ-MM-TT oder TT.MM.JJJJ)", "datum"),
            ("Menge (g)", "menge"),
            ("Bearbeiter", "bearbeiter"),
            ("Kommentar", "kommentar"),
        ]
        entries = {}
        for idx, (label, key) in enumerate(fields):
            tk.Label(dlg, text=label+":", anchor="w").grid(row=idx, column=0, sticky="w", padx=8, pady=4)
            entry = tk.Entry(dlg, width=26)
            entry.grid(row=idx, column=1, pady=4)
            entries[key] = entry

        def save():
            data = {k: v.get().strip() for k, v in entries.items()}
            try:
                data["mitglied_id"] = int(data["mitglied_id"])
                data["charge_id"] = int(data["charge_id"])
                data["menge"] = float(data["menge"])
            except Exception:
                messagebox.showerror("Fehler", "Mitglied-ID, Charge-ID und Menge müssen Zahlen sein.")
                return
            # Datum flexibel parsen
            try:
                data["datum"] = parse_datum(data["datum"])
            except Exception:
                messagebox.showerror("Fehler", "Bitte Datum im Format JJJJ-MM-TT oder TT.MM.JJJJ eingeben!")
                return
            # Limit prüfen
            monat = data["datum"][:7]  # "YYYY-MM"
            bisher = model.get_monatsmenge_fuer_mitglied(data["mitglied_id"], monat)
            limit = settings_model.get_limit()
            verbleibend = limit - bisher
            if data["menge"] > verbleibend:
                messagebox.showwarning(
                    "Limit überschritten",
                    f"Limit für diesen Monat ist {limit}g.\n"
                    f"Bisher gebucht: {bisher:.2f}g\n"
                    f"Verbleibend: {verbleibend:.2f}g\n"
                    f"Abgabe nicht möglich."
                )
                return
            # Abgabe speichern
            model.add_abgabe(data)
            messagebox.showinfo("Erfolg", "Abgabe verbucht.")
            dlg.destroy()
            refresh_abgaben()
            refresh_chargen()

        tk.Button(dlg, text="Speichern", command=save).grid(row=len(fields), column=0, pady=20)
        tk.Button(dlg, text="Abbrechen", command=dlg.destroy).grid(row=len(fields), column=1, pady=20)

    tk.Button(win, text="Neue Abgabe", command=open_add_abgabe).pack(pady=4)

    # --- Abgabe stornieren/löschen ---
    def delete_selected_abgabe():
        sel = tree_a.selection()
        if not sel:
            messagebox.showinfo("Info", "Bitte eine Abgabe auswählen.")
            return
        abgabe_id = tree_a.item(sel[0])["values"][0]
        if messagebox.askyesno("Stornieren", "Abgabe wirklich stornieren und Menge zurückbuchen?"):
            model.delete_abgabe(abgabe_id)
            messagebox.showinfo("Erfolg", "Abgabe storniert.")
            refresh_abgaben()
            refresh_chargen()

    tk.Button(win, text="Abgabe stornieren", command=delete_selected_abgabe).pack(pady=4)
    tk.Button(win, text="Schließen", command=win.destroy).pack(pady=15)