import sqlite3
import os

DB_PATH = os.path.join(
    os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
    "db",
    "verein.db"
)

def init_db():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        # Chargen-Tabelle
        c.execute("""
            CREATE TABLE IF NOT EXISTS chargen (
                charge_id INTEGER PRIMARY KEY AUTOINCREMENT,
                bezeichnung TEXT,
                start_datum TEXT,
                end_datum TEXT,
                menge_gesamt REAL,
                menge_rest REAL,
                kommentar TEXT
            );
        """)
        # Abgaben-Tabelle
        c.execute("""
            CREATE TABLE IF NOT EXISTS abgaben (
                abgabe_id INTEGER PRIMARY KEY AUTOINCREMENT,
                mitglied_id INTEGER,
                charge_id INTEGER,
                datum TEXT,
                menge REAL,
                bearbeiter TEXT,
                kommentar TEXT,
                FOREIGN KEY(mitglied_id) REFERENCES mitglieder(mitglied_id),
                FOREIGN KEY(charge_id) REFERENCES chargen(charge_id)
            );
        """)
        conn.commit()

def get_chargen():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("SELECT charge_id, bezeichnung, start_datum, menge_gesamt, menge_rest FROM chargen ORDER BY start_datum DESC;")
        return c.fetchall()

def add_charge(data):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            INSERT INTO chargen (bezeichnung, start_datum, end_datum, menge_gesamt, menge_rest, kommentar)
            VALUES (?, ?, ?, ?, ?, ?);
        """, (
            data['bezeichnung'],
            data['start_datum'],
            data['end_datum'],
            data['menge_gesamt'],
            data['menge_gesamt'],
            data['kommentar']
        ))
        conn.commit()

def update_charge(charge_id, data):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            UPDATE chargen SET
                bezeichnung = ?,
                start_datum = ?,
                end_datum = ?,
                menge_gesamt = ?,
                menge_rest = ?,
                kommentar = ?
            WHERE charge_id = ?
        """, (
            data['bezeichnung'],
            data['start_datum'],
            data['end_datum'],
            data['menge_gesamt'],
            data['menge_rest'],
            data['kommentar'],
            charge_id
        ))
        conn.commit()

def delete_charge(charge_id):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        # Optional: Vor dem Löschen alle zugehörigen Abgaben löschen oder verhindern!
        c.execute("DELETE FROM chargen WHERE charge_id = ?", (charge_id,))
        conn.commit()

def get_abgaben(mitglied_id=None, monat=None):
    """
    Gibt alle Abgaben zurück:
    - für ein Mitglied (wenn mitglied_id gesetzt)
    - und/oder nur für einen Monat (Format 'YYYY-MM')
    Rückgabe: Liste von Tupeln (abgabe_id, mitglied_id, charge_id, datum, menge, bearbeiter, kommentar)
    """
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        if mitglied_id and monat:
            c.execute("""
                SELECT abgabe_id, mitglied_id, charge_id, datum, menge, bearbeiter, kommentar
                FROM abgaben
                WHERE mitglied_id = ? AND datum LIKE ?
                ORDER BY datum DESC
            """, (mitglied_id, f"{monat}%"))
        elif mitglied_id:
            c.execute("""
                SELECT abgabe_id, mitglied_id, charge_id, datum, menge, bearbeiter, kommentar
                FROM abgaben
                WHERE mitglied_id = ?
                ORDER BY datum DESC
            """, (mitglied_id,))
        else:
            c.execute("""
                SELECT abgabe_id, mitglied_id, charge_id, datum, menge, bearbeiter, kommentar
                FROM abgaben
                ORDER BY datum DESC
            """)
        return c.fetchall()

def add_abgabe(data):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            INSERT INTO abgaben (mitglied_id, charge_id, datum, menge, bearbeiter, kommentar)
            VALUES (?, ?, ?, ?, ?, ?)
        """, (
            data['mitglied_id'],
            data['charge_id'],
            data['datum'],
            data['menge'],
            data['bearbeiter'],
            data['kommentar']
        ))
        # Menge von Charge aktualisieren
        c.execute(
            "UPDATE chargen SET menge_rest = menge_rest - ? WHERE charge_id = ?",
            (data['menge'], data['charge_id'])
        )
        conn.commit()

def update_abgabe(abgabe_id, data):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            UPDATE abgaben SET
                mitglied_id = ?,
                charge_id = ?,
                datum = ?,
                menge = ?,
                bearbeiter = ?,
                kommentar = ?
            WHERE abgabe_id = ?
        """, (
            data['mitglied_id'],
            data['charge_id'],
            data['datum'],
            data['menge'],
            data['bearbeiter'],
            data['kommentar'],
            abgabe_id
        ))
        conn.commit()

def delete_abgabe(abgabe_id):
    """Storniert (löscht) eine Abgabe und bucht die Menge zurück auf die Charge."""
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        # Hole Menge und charge_id der Abgabe
        c.execute("SELECT menge, charge_id FROM abgaben WHERE abgabe_id = ?", (abgabe_id,))
        row = c.fetchone()
        if row:
            menge, charge_id = row
            # Menge zurückbuchen
            c.execute("UPDATE chargen SET menge_rest = menge_rest + ? WHERE charge_id = ?", (menge, charge_id))
            # Abgabe löschen
            c.execute("DELETE FROM abgaben WHERE abgabe_id = ?", (abgabe_id,))
            conn.commit()

def get_monatsmenge_fuer_mitglied(mitglied_id, monat):
    """
    Gesamtmenge (g) für ein Mitglied in einem bestimmten Monat.
    :param mitglied_id: int
    :param monat: str, Format "YYYY-MM"
    """
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            SELECT SUM(menge) FROM abgaben
            WHERE mitglied_id = ? AND datum LIKE ?
        """, (mitglied_id, f"{monat}%"))
        res = c.fetchone()
        return float(res[0]) if res[0] else 0.0