import sqlite3
import os
import datetime

DB_PATH = os.path.join(
    os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
    "db",
    "verein.db"
)

def init_db():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            CREATE TABLE IF NOT EXISTS mitglieder (
                mitglied_id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT,
                vorname TEXT,
                geburtsdatum TEXT,
                adresse TEXT,
                eintrittsdatum TEXT,
                status TEXT DEFAULT 'aktiv',
                mitgliedsnummer TEXT UNIQUE,
                barcode_id TEXT,
                nfc_uid TEXT,
                email TEXT,
                telefon TEXT
            );
        """)
        conn.commit()

def get_alle():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("SELECT mitglied_id, vorname, name, status, mitgliedsnummer FROM mitglieder ORDER BY name, vorname;")
        return c.fetchall()

def get_alle_full():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("SELECT mitglied_id, vorname, name, status, mitgliedsnummer, email FROM mitglieder ORDER BY name, vorname;")
        return c.fetchall()

def next_mitgliedsnummer():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("SELECT MAX(CAST(mitgliedsnummer AS INTEGER)) FROM mitglieder")
        maxnum = c.fetchone()[0]
        if not maxnum or not str(maxnum).isdigit():
            # Start mit aktuellem Jahr und 0001, z.B. 20250001
            return f"{datetime.datetime.now().year}0001"
        return str(int(maxnum) + 1)

def add_mitglied(data):
    if not data.get("mitgliedsnummer"):
        data["mitgliedsnummer"] = next_mitgliedsnummer()
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            INSERT INTO mitglieder
            (name, vorname, geburtsdatum, adresse, eintrittsdatum, status, mitgliedsnummer, barcode_id, nfc_uid, email, telefon)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, (
            data['name'], data['vorname'], data['geburtsdatum'], data['adresse'], data['eintrittsdatum'],
            data['status'], data['mitgliedsnummer'], data['barcode_id'], data['nfc_uid'], data['email'], data['telefon']
        ))
        conn.commit()

def get_by_id(mitglied_id):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("SELECT * FROM mitglieder WHERE mitglied_id = ?", (mitglied_id,))
        row = c.fetchone()
        if not row:
            return {}
        keys = [d[0] for d in c.description]
        return dict(zip(keys, row))

def update_mitglied(mitglied_id, data):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("""
            UPDATE mitglieder SET
                vorname = ?, name = ?, geburtsdatum = ?, adresse = ?, eintrittsdatum = ?,
                status = ?, mitgliedsnummer = ?, barcode_id = ?, nfc_uid = ?, email = ?, telefon = ?
            WHERE mitglied_id = ?
        """, (
            data['vorname'], data['name'], data['geburtsdatum'], data['adresse'], data['eintrittsdatum'],
            data['status'], data['mitgliedsnummer'], data['barcode_id'], data['nfc_uid'], data['email'], data['telefon'], mitglied_id
        ))
        conn.commit()

def delete_mitglied(mitglied_id):
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        c.execute("DELETE FROM mitglieder WHERE mitglied_id = ?", (mitglied_id,))
        conn.commit()