import os
import sqlite3

# Absoluter Pfad zum Projekt-Root (wo main.py liegt)
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
DB_DIR = os.path.join(BASE_DIR, "db")
DB_PATH = os.path.join(DB_DIR, "verein.db")

# Schritt 1: db-Ordner anlegen (falls nicht vorhanden)
if not os.path.exists(DB_DIR):
    os.makedirs(DB_DIR)
    print(f"Verzeichnis '{DB_DIR}' wurde angelegt.")
else:
    print(f"Verzeichnis '{DB_DIR}' existiert bereits.")

# Schritt 2: Datenbank und Grundtabellen anlegen
def init_db():
    with sqlite3.connect(DB_PATH) as conn:
        c = conn.cursor()
        # Tabellen für Einstellungen, Mitglieder, Kassenbuch, Staff, Chargen, Abgaben (Minimalausführung)
        c.execute("""
            CREATE TABLE IF NOT EXISTS verein_settings (
                key TEXT PRIMARY KEY,
                value TEXT
            );
        """)
        c.execute("""
            CREATE TABLE IF NOT EXISTS mitglieder (
                mitglied_id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT,
                vorname TEXT,
                geburtsdatum TEXT,
                adresse TEXT,
                eintrittsdatum TEXT,
                status TEXT DEFAULT 'aktiv',
                mitgliedsnummer TEXT UNIQUE,
                barcode_id TEXT,
                nfc_uid TEXT,
                email TEXT,
                telefon TEXT
            );
        """)
        c.execute("""
            CREATE TABLE IF NOT EXISTS kassenbuch (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                datum TEXT,
                art TEXT,
                kategorie TEXT,
                betrag REAL,
                zweck TEXT,
                beleg TEXT,
                buchende_person TEXT,
                timestamp DATETIME DEFAULT CURRENT_TIMESTAMP
            );
        """)
        c.execute("""
            CREATE TABLE IF NOT EXISTS staff_accounts (
                staff_id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT UNIQUE NOT NULL,
                fullname TEXT,
                email TEXT,
                role TEXT NOT NULL,
                status TEXT DEFAULT 'active',
                password_hash TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME,
                last_login DATETIME
            );
        """)
        c.execute("""
            CREATE TABLE IF NOT EXISTS chargen (
                charge_id INTEGER PRIMARY KEY AUTOINCREMENT,
                bezeichnung TEXT,
                start_datum TEXT,
                end_datum TEXT,
                menge_gesamt REAL,
                menge_rest REAL,
                kommentar TEXT
            );
        """)
        c.execute("""
            CREATE TABLE IF NOT EXISTS abgaben (
                abgabe_id INTEGER PRIMARY KEY AUTOINCREMENT,
                mitglied_id INTEGER,
                charge_id INTEGER,
                datum TEXT,
                menge REAL,
                bearbeiter TEXT,
                kommentar TEXT,
                FOREIGN KEY(mitglied_id) REFERENCES mitglieder(mitglied_id),
                FOREIGN KEY(charge_id) REFERENCES chargen(charge_id)
            );
        """)
        # Standardwerte für Settings
        for k, v in [
            ("name", ""),
            ("id", ""),
            ("adresse", ""),
            ("email", ""),
            ("bank", ""),
            ("abgabe_limit", "25"),
        ]:
            c.execute("INSERT OR IGNORE INTO verein_settings (key, value) VALUES (?, ?)", (k, v))
        conn.commit()
    print(f"Datenbank '{DB_PATH}' und Grundtabellen wurden eingerichtet.")

if __name__ == "__main__":
    init_db()
    print("Setup abgeschlossen! Du kannst jetzt main.py starten.")
