// PKZIP Core Functions - CRC32 and Key Updates
// For Tang Nano 4K FPGA Password Cracker

// ============================================================================
// CRC32 LOOKUP TABLE MODULE
// Stored in Block RAM for efficient access
// ============================================================================

module crc32_table (
    input  wire        clk,
    input  wire [7:0]  index,
    output reg  [31:0] value
);

reg [31:0] crc_table [0:255];

// Variables for table generation
integer i, j;
reg [31:0] c;

// Pre-compute CRC32 table at synthesis time
initial begin
    for (i = 0; i < 256; i = i + 1) begin
        c = i;
        for (j = 0; j < 8; j = j + 1) begin
            if (c[0])
                c = (c >> 1) ^ 32'hEDB88320;
            else
                c = c >> 1;
        end
        crc_table[i] = c;
    end
end

always @(posedge clk) begin
    value <= crc_table[index];
end

endmodule

// ============================================================================
// PKZIP KEY UPDATE ENGINE
// Updates the 3 PKZIP encryption keys with one password byte
// ============================================================================

module pkzip_key_update (
    input  wire        clk,
    input  wire        rst_n,
    input  wire        start,
    input  wire [7:0]  pwd_byte,
    input  wire [31:0] key0_in,
    input  wire [31:0] key1_in,
    input  wire [31:0] key2_in,
    output reg  [31:0] key0_out,
    output reg  [31:0] key1_out,
    output reg  [31:0] key2_out,
    output reg         done
);

reg [2:0] state;  // Changed to 3 bits for extra state
reg [31:0] temp;
wire [31:0] crc_val;
reg [7:0] crc_idx;

crc32_table crc_inst (
    .clk(clk),
    .index(crc_idx),
    .value(crc_val)
);

localparam IDLE  = 3'd0;
localparam KEY0  = 3'd1;
localparam KEY1  = 3'd2;
localparam KEY1_WAIT = 3'd3;  // NEW: Wait for key1 to be written
localparam KEY2  = 3'd4;
localparam KEY2_DONE = 3'd5;  // NEW: Finish key2

always @(posedge clk or negedge rst_n) begin
    if (!rst_n) begin
        state <= IDLE;
        done <= 1'b0;
        key0_out <= 32'h12345678;
        key1_out <= 32'h23456789;
        key2_out <= 32'h34567890;
    end else begin
        case (state)
            IDLE: begin
                done <= 1'b0;
                if (start) begin
                    key0_out <= key0_in;
                    key1_out <= key1_in;
                    key2_out <= key2_in;
                    state <= KEY0;
                end
            end
            
            KEY0: begin
                // Update key0: CRC32(key0, pwd_byte)
                crc_idx <= (key0_out ^ pwd_byte) & 8'hFF;
                temp <= key0_out >> 8;
                state <= KEY1;
            end
            
            KEY1: begin
                // Complete key0 update
                key0_out <= temp ^ crc_val;
                
                // Update key1: (key1 + (key0 & 0xFF)) * 0x08088405 + 1
                // Use the NEW key0 value (temp ^ crc_val)
                temp <= key1_out + ((temp ^ crc_val) & 32'h000000FF);
                state <= KEY1_WAIT;
            end
            
            KEY1_WAIT: begin
                // Complete key1 update
                key1_out <= (temp * 32'h08088405) + 32'h00000001;
                state <= KEY2;
            end
            
            KEY2: begin
                // Now use the NEW key1_out for key2 update
                // Update key2: CRC32(key2, key1 >> 24)
                crc_idx <= (key2_out ^ key1_out[31:24]) & 8'hFF;
                temp <= key2_out >> 8;
                state <= KEY2_DONE;
            end
            
            KEY2_DONE: begin
                // Complete key2 update
                key2_out <= temp ^ crc_val;
                done <= 1'b1;
                state <= IDLE;
            end
            
            default: begin
                // Should never happen, but needed for synthesis
                state <= IDLE;
                done <= 1'b0;
            end
        endcase
    end
end

endmodule

// ============================================================================
// PASSWORD TO CHARSET INDEX CONVERTER
// Converts password index to actual characters based on charset
// ============================================================================

module password_generator (
    input  wire [2:0]  charset,      // Charset selection
    input  wire [3:0]  pwd_len,      // Password length (1-8)
    input  wire [31:0] pwd_index,    // Password index to generate
    output wire [63:0] pwd_packed    // Output as packed 8x8bit = 64bit
);

// Charset definitions
localparam CHARSET_NUMERIC = 3'd0;  // 0-9
localparam CHARSET_LOWER   = 3'd1;  // a-z
localparam CHARSET_UPPER   = 3'd2;  // A-Z
localparam CHARSET_ALPHA   = 3'd3;  // a-zA-Z
localparam CHARSET_ALNUM   = 3'd4;  // a-zA-Z0-9
localparam CHARSET_SPECIAL = 3'd5;  // ASCII 32-95
localparam CHARSET_ASCII   = 3'd6;  // ASCII 32-126

function [7:0] get_charset_size;
    input [2:0] cs;
    begin
        case (cs)
            CHARSET_NUMERIC: get_charset_size = 8'd10;
            CHARSET_LOWER:   get_charset_size = 8'd26;
            CHARSET_UPPER:   get_charset_size = 8'd26;
            CHARSET_ALPHA:   get_charset_size = 8'd52;
            CHARSET_ALNUM:   get_charset_size = 8'd62;
            CHARSET_SPECIAL: get_charset_size = 8'd64;
            CHARSET_ASCII:   get_charset_size = 8'd95;
            default:         get_charset_size = 8'd26;
        endcase
    end
endfunction

function [7:0] index_to_char;
    input [2:0] cs;
    input [7:0] idx;
    begin
        case (cs)
            CHARSET_NUMERIC: begin
                index_to_char = 8'h30 + idx;  // '0'-'9'
            end
            CHARSET_LOWER: begin
                index_to_char = 8'h61 + idx;  // 'a'-'z'
            end
            CHARSET_UPPER: begin
                index_to_char = 8'h41 + idx;  // 'A'-'Z'
            end
            CHARSET_ALPHA: begin
                if (idx < 26)
                    index_to_char = 8'h61 + idx;        // 'a'-'z'
                else
                    index_to_char = 8'h41 + idx[7:0] - 8'd26; // 'A'-'Z'
            end
            CHARSET_ALNUM: begin
                if (idx < 26)
                    index_to_char = 8'h61 + idx;        // 'a'-'z'
                else if (idx < 52)
                    index_to_char = 8'h41 + idx[7:0] - 8'd26; // 'A'-'Z'
                else
                    index_to_char = 8'h30 + idx[7:0] - 8'd52; // '0'-'9'
            end
            CHARSET_SPECIAL: begin
                index_to_char = 8'h20 + idx;  // ASCII 32-95
            end
            CHARSET_ASCII: begin
                index_to_char = 8'h20 + idx;  // ASCII 32-126
            end
            default: begin
                index_to_char = 8'h61 + idx;  // Default to lowercase
            end
        endcase
    end
endfunction

// Convert password index to characters
reg [7:0] pwd_chars [0:7];
integer k;
reg [31:0] temp_idx;
reg [7:0] cs_size;
reg [7:0] digit;

always @(*) begin
    cs_size = get_charset_size(charset);
    temp_idx = pwd_index;
    digit = 8'h00;  // Initialize to prevent latch
    
    for (k = 0; k < 8; k = k + 1) begin
        if (k < pwd_len) begin
            digit = temp_idx % cs_size;
            pwd_chars[k] = index_to_char(charset, digit);
            temp_idx = temp_idx / cs_size;
        end else begin
            pwd_chars[k] = 8'h00;  // Null for unused positions
        end
    end
end

// Pack into 64-bit output
assign pwd_packed = {pwd_chars[7], pwd_chars[6], pwd_chars[5], pwd_chars[4],
                     pwd_chars[3], pwd_chars[2], pwd_chars[1], pwd_chars[0]};

endmodule
