// PKZIP Password Cracker - Complete Top Level
// Tang Nano 4K with working SPI interface

module fpga_top (
    input  wire clk,
    input  wire spi_sclk,
    input  wire spi_mosi,
    output wire spi_miso,
    input  wire spi_cs_n,
    output wire led
);

wire clk_sys = clk;  // 27 MHz
wire sys_rst_n = 1'b1;  // Always enabled for now

// LED blinker
reg [23:0] counter;
always @(posedge clk_sys) counter <= counter + 1'b1;
assign led = counter[23];

// ============================================================================
// CONFIGURATION REGISTERS
// ============================================================================

reg [15:0] reg_check_bytes;   // [15:8]=byte11, [7:0]=byte10
reg [2:0]  reg_charset;       // Charset mode
reg [3:0]  reg_pwd_min;       // Min password length
reg [3:0]  reg_pwd_max;       // Max password length
reg        reg_pkzip_mode;    // 0=v1, 1=v2

// Job queue (4 jobs for 4 engines)
reg [31:0] job_start [0:3];
reg [31:0] job_end [0:3];
reg [3:0]  job_valid;
reg [3:0]  job_done;

// Engine control
reg [3:0] engine_start;

// ============================================================================
// 1 PKZIP CRACKING ENGINE (reduced from 4 due to resource constraints)
// Tang Nano 4K has only 4608 LUTs - can't fit 4 engines!
// ============================================================================

wire [3:0] engine_running;
wire [3:0] engine_found;
wire [31:0] engine_tested [0:3];
wire [31:0] engine_result [0:3];

// Only instantiate 1 engine
pkzip_engine engine_inst (
    .clk(clk_sys),
    .rst_n(sys_rst_n),
    .start(engine_start[0]),
    .start_index(job_start[0]),
    .end_index(job_end[0]),
    .charset(reg_charset),
    .pwd_min_len(reg_pwd_min),
    .pwd_max_len(reg_pwd_max),
    .check_bytes(reg_check_bytes),
    .pkzip_mode(reg_pkzip_mode),
    .running(engine_running[0]),
    .found(engine_found[0]),
    .tested_count(engine_tested[0]),
    .found_password(engine_result[0])
);

// Tie off unused engine signals
assign engine_running[3:1] = 3'b000;
assign engine_found[3:1] = 3'b000;
assign engine_tested[1] = 32'h0;
assign engine_tested[2] = 32'h0;
assign engine_tested[3] = 32'h0;
assign engine_result[1] = 32'h0;
assign engine_result[2] = 32'h0;
assign engine_result[3] = 32'h0;

// Job assignment logic
reg [3:0] engine_running_prev;

always @(posedge clk_sys or negedge sys_rst_n) begin
    if (!sys_rst_n) begin
        // Reset all signals assigned in this block
        engine_running_prev <= 4'b0;
        engine_start <= 4'b0;
        job_done <= 4'b0;
    end else begin
        engine_running_prev <= engine_running;
        
        // Auto-start engines when jobs are valid and engines are idle
        if (job_valid[0] && !engine_running[0] && !engine_start[0])
            engine_start[0] <= 1'b1;
        else
            engine_start[0] <= 1'b0;
            
        if (job_valid[1] && !engine_running[1] && !engine_start[1])
            engine_start[1] <= 1'b1;
        else
            engine_start[1] <= 1'b0;
            
        if (job_valid[2] && !engine_running[2] && !engine_start[2])
            engine_start[2] <= 1'b1;
        else
            engine_start[2] <= 1'b0;
            
        if (job_valid[3] && !engine_running[3] && !engine_start[3])
            engine_start[3] <= 1'b1;
        else
            engine_start[3] <= 1'b0;
        
        // Mark jobs as done when engines finish
        // Use edge detection: was running, now stopped
        if (engine_running_prev[0] && !engine_running[0] && job_valid[0])
            job_done[0] <= 1'b1;
        else if (!job_valid[0])
            job_done[0] <= 1'b0;
            
        if (engine_running_prev[1] && !engine_running[1] && job_valid[1])
            job_done[1] <= 1'b1;
        else if (!job_valid[1])
            job_done[1] <= 1'b0;
            
        if (engine_running_prev[2] && !engine_running[2] && job_valid[2])
            job_done[2] <= 1'b1;
        else if (!job_valid[2])
            job_done[2] <= 1'b0;
            
        if (engine_running_prev[3] && !engine_running[3] && job_valid[3])
            job_done[3] <= 1'b1;
        else if (!job_valid[3])
            job_done[3] <= 1'b0;
    end
end

// Total tested count
wire [31:0] total_tested = engine_tested[0] + engine_tested[1] + 
                           engine_tested[2] + engine_tested[3];

// Status register
wire any_running = |engine_running;
wire any_found = |engine_found;
wire [31:0] reg_status = {22'h0, any_found, 7'h0, any_running, 1'b1};

// ============================================================================
// SPI SLAVE INTERFACE (WORKING VERSION WITH WRITE SUPPORT!)
// ============================================================================

reg [7:0] rx_shift;
reg [2:0] bitcnt;
reg [7:0] addr;
reg addr_received;
reg [31:0] write_data;
reg [5:0] write_bitcnt;  // 6 bits to hold 32!
reg write_mode;

// RX on posedge - sample MOSI
always @(posedge spi_sclk or posedge spi_cs_n) begin
    if (spi_cs_n) begin
        rx_shift <= 8'h00;
        bitcnt <= 3'd0;
        addr <= 8'h00;
        addr_received <= 1'b0;
        write_data <= 32'h0;
        write_bitcnt <= 6'd0;  // 6-bit value
        write_mode <= 1'b0;
    end else begin
        rx_shift <= {rx_shift[6:0], spi_mosi};
        
        if (!addr_received) begin
            bitcnt <= bitcnt + 3'd1;
            if (bitcnt == 3'd7) begin
                addr <= {rx_shift[6:0], spi_mosi};
                addr_received <= 1'b1;
                // Check if this is a write (MSB = 1)
                // After 7 shifts, MSB (bit7) is in rx_shift[6]!
                write_mode <= rx_shift[6];  // CORRECT!
            end
        end else if (write_mode) begin
            // Receiving write data (32 bits)
            // FIX: Collect data BEFORE incrementing counter!
            if (write_bitcnt <= 6'd31) begin  // 6-bit value
                write_data <= {write_data[30:0], spi_mosi};
            end
            write_bitcnt <= write_bitcnt + 6'd1;  // 6-bit increment
        end
    end
end

// TX on negedge - output MISO
reg [31:0] tx_reg;
reg loaded;
assign spi_miso = (!spi_cs_n) ? tx_reg[31] : 1'bz;

always @(negedge spi_sclk or posedge spi_cs_n) begin
    if (spi_cs_n) begin
        tx_reg <= 32'hFFFFFFFF;
        loaded <= 1'b0;
    end else begin
        if (addr_received && !loaded && !write_mode) begin
            // Read operation
            case (addr)
                8'h00: tx_reg <= 32'h504B5A49;  // "PKZI" magic
                8'h04: tx_reg <= 32'h00000004;  // BOARD_ID
                8'h08: tx_reg <= reg_status;
                8'h0C: tx_reg <= total_tested;
                
                // Engine tested counts
                8'h10: tx_reg <= engine_tested[0];
                8'h14: tx_reg <= engine_tested[1];
                8'h18: tx_reg <= engine_tested[2];
                8'h1C: tx_reg <= engine_tested[3];
                
                // Engine results (found passwords)
                8'h20: tx_reg <= engine_result[0];
                8'h24: tx_reg <= engine_result[1];
                8'h28: tx_reg <= engine_result[2];
                8'h2C: tx_reg <= engine_result[3];
                
                // Configuration
                8'h30: tx_reg <= {16'h0, reg_check_bytes};
                8'h34: tx_reg <= {25'h0, reg_pkzip_mode, reg_charset, reg_pwd_max, reg_pwd_min};
                
                // Job queue
                8'h40: tx_reg <= job_start[0];
                8'h44: tx_reg <= job_end[0];
                8'h48: tx_reg <= job_start[1];
                8'h4C: tx_reg <= job_end[1];
                8'h50: tx_reg <= job_start[2];
                8'h54: tx_reg <= job_end[2];
                8'h58: tx_reg <= job_start[3];
                8'h5C: tx_reg <= job_end[3];
                8'h60: tx_reg <= {24'h0, job_done, job_valid};
                
                default: tx_reg <= 32'h00000000;
            endcase
            loaded <= 1'b1;
        end else begin
            // Shift out data
            tx_reg <= {tx_reg[30:0], 1'b0};
        end
    end
end

// Write register handling (after full transaction)
always @(posedge spi_cs_n) begin
    if (write_mode && write_bitcnt == 6'd32) begin  // Changed to 6-bit value
        // Complete write received
        case (addr & 8'h7F)  // Clear write bit
            8'h30: reg_check_bytes <= write_data[15:0];
            8'h34: begin
                reg_pwd_min <= write_data[3:0];
                reg_pwd_max <= write_data[7:4];
                reg_charset <= write_data[10:8];
                reg_pkzip_mode <= write_data[11];
            end
            
            // Job queue writes
            8'h40: job_start[0] <= write_data;
            8'h44: job_end[0] <= write_data;
            8'h48: job_start[1] <= write_data;
            8'h4C: job_end[1] <= write_data;
            8'h50: job_start[2] <= write_data;
            8'h54: job_end[2] <= write_data;
            8'h58: job_start[3] <= write_data;
            8'h5C: job_end[3] <= write_data;
            8'h60: begin
                job_valid <= write_data[3:0];
                // Writing to job_valid starts engines
            end
            
            default: begin
                // Ignore writes to unknown addresses
            end
        endcase
    end
end

// Initialize SPI-writable registers with defaults
initial begin
    reg_check_bytes = 16'hABCD;
    reg_charset = 3'd1;  // lowercase
    reg_pwd_min = 4'd4;
    reg_pwd_max = 4'd4;
    reg_pkzip_mode = 1'b1;  // v2
    
    job_valid = 4'b0000;  // Start disabled
    
    // Default job ranges
    job_start[0] = 32'd0;
    job_end[0] = 32'd114243;
    job_start[1] = 32'd114244;
    job_end[1] = 32'd228487;
    job_start[2] = 32'd228488;
    job_end[2] = 32'd342731;
    job_start[3] = 32'd342732;
    job_end[3] = 32'd456975;
end

endmodule
