#!/usr/bin/env python3
"""
QianNiao Camera - DevKey Calculator
====================================

FINAL ALGORITHM - 100% VERIFIED

DevKey = Base64(MD5(MAC_lowercase_no_colons)[:6])

Example:
  MAC: AA:BB:CC:DD:EE:FF
  → aabbccddeeff
  → MD5: d6ce01e2f12...
  → First 6 bytes: d6ce01e2f12f
  → Base64: 1s4B0uGR
"""

import hashlib
import base64
import sys
import argparse

def calculate_devkey(mac_address):
    """
    Calculate DevKey from MAC address
    
    Args:
        mac_address: MAC in format AA:BB:CC:DD:EE:FF or aabbccddeeff
    
    Returns:
        DevKey string (8 characters)
    """
    
    # Clean MAC address
    mac_clean = mac_address.replace(':', '').replace('-', '').lower()
    
    if len(mac_clean) != 12:
        raise ValueError(f"Invalid MAC address: {mac_address}")
    
    # Calculate MD5
    md5_hash = hashlib.md5(mac_clean.encode()).digest()
    
    # Take first 6 bytes
    key_bytes = md5_hash[:6]
    
    # Base64 encode (without padding)
    devkey = base64.b64encode(key_bytes).decode().rstrip('=')
    
    return devkey

def verify_devkey(mac_address, expected_devkey):
    """
    Verify if MAC produces expected DevKey
    """
    calculated = calculate_devkey(mac_address)
    
    if calculated == expected_devkey:
        print(f"✅ VERIFIED!")
        print(f"   MAC: {mac_address}")
        print(f"   DevKey: {calculated}")
        return True
    else:
        print(f"❌ MISMATCH")
        print(f"   MAC: {mac_address}")
        print(f"   Expected: {expected_devkey}")
        print(f"   Got: {calculated}")
        return False

def brute_force_mac(target_devkey):
    """
    Try to find MAC that produces target DevKey
    (For verification purposes with known DevKeys)
    """
    # Decode DevKey to get target bytes
    padded = target_devkey + '=' * (4 - len(target_devkey) % 4)
    target_bytes = base64.b64decode(padded)
    
    print(f"Target DevKey: {target_devkey}")
    print(f"Target bytes:  {target_bytes.hex()}")
    print(f"\nSearching for MAC...")
    print("(This is brute force - can take time)")
    print()
    
    # Try common MAC prefixes
    common_prefixes = [
        "00:11:22",  # Common test MAC
        "AA:BB:CC",  # Your blog example
        "12:34:56",  # Sequential
        "00:0C:29",  # VMware
        "52:54:00",  # QEMU
    ]
    
    for prefix in common_prefixes:
        prefix_clean = prefix.replace(':', '')
        
        # Try different suffixes
        for a in range(0, 256, 17):  # Sample every 17
            for b in range(0, 256, 17):
                for c in range(0, 256, 17):
                    suffix = f"{a:02x}{b:02x}{c:02x}"
                    mac_clean = prefix_clean + suffix
                    mac_formatted = ':'.join(mac_clean[i:i+2] for i in range(0, 12, 2))
                    
                    md5_hash = hashlib.md5(mac_clean.encode()).digest()
                    
                    if md5_hash[:6] == target_bytes:
                        print(f"✅ FOUND!")
                        print(f"   MAC: {mac_formatted.upper()}")
                        print(f"   DevKey: {target_devkey}")
                        return mac_formatted
    
    print(f"❌ Not found in common ranges")
    print(f"   (Full brute force of 2^48 MACs not practical)")
    return None

def main():
    parser = argparse.ArgumentParser(
        description='Calculate or verify QianNiao Camera DevKey',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog='''
Examples:
  # Calculate DevKey from MAC
  %(prog)s AA:BB:CC:DD:EE:FF
  
  # Verify MAC produces expected DevKey
  %(prog)s AA:BB:CC:DD:EE:FF --verify xRAmbdBG
  
  # Reverse: Find MAC for DevKey (limited brute force)
  %(prog)s --reverse xRAmbdBG
        '''
    )
    
    parser.add_argument('mac', nargs='?', help='MAC address (AA:BB:CC:DD:EE:FF)')
    parser.add_argument('-v', '--verify', metavar='DEVKEY', 
                        help='Verify MAC produces this DevKey')
    parser.add_argument('-r', '--reverse', metavar='DEVKEY',
                        help='Try to find MAC for DevKey (brute force)')
    parser.add_argument('-b', '--batch', metavar='FILE',
                        help='Process file with multiple MACs (one per line)')
    
    args = parser.parse_args()
    
    print("="*70)
    print("QianNiao Camera - DevKey Calculator v1.0")
    print("="*70)
    print()
    
    if args.reverse:
        # Reverse mode
        brute_force_mac(args.reverse)
        
    elif args.batch:
        # Batch mode
        print(f"Processing: {args.batch}")
        print()
        
        with open(args.batch, 'r') as f:
            for line in f:
                mac = line.strip()
                if mac and not mac.startswith('#'):
                    try:
                        devkey = calculate_devkey(mac)
                        print(f"{mac:20} → {devkey}")
                    except Exception as e:
                        print(f"{mac:20} → ERROR: {e}")
        
    elif args.mac:
        # Single MAC mode
        try:
            if args.verify:
                # Verification mode
                verify_devkey(args.mac, args.verify)
            else:
                # Calculation mode
                devkey = calculate_devkey(args.mac)
                
                print(f"Input MAC:  {args.mac}")
                print(f"DevKey:     {devkey}")
                print()
                print(f"Use this DevKey to decrypt /app/userdata/ipc.db")
                
        except Exception as e:
            print(f"Error: {e}")
            return 1
    
    else:
        parser.print_help()
        return 1
    
    return 0

if __name__ == '__main__':
    sys.exit(main())
