#!/bin/sh
################################################################################
#
# QianNiao Camera - Firmware Flash Tool
# 
# Copyright (c) 2025 datenretter-pro.de
# Version: 3.0 FINAL
#
# Flasht vorgefertigtes Firmware-Image (ultimate_mtd6.img)
# Schnelle Alternative zum Full-Unlock-Script
#
################################################################################

VERSION="3.0"
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m'

print_banner() {
    clear
    echo ""
    echo -e "${CYAN}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║${NC}  ${BOLD}QianNiao Firmware Flash Tool${NC}                                   ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}  Version ${VERSION} - datenretter-pro.de                             ${CYAN}║${NC}"
    echo -e "${CYAN}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo -e "${BOLD}${CYAN}▶ $1${NC}"
}

print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_banner

# Check root
if [ "$(id -u)" -ne 0 ]; then
    print_error "Muss als root ausgeführt werden!"
    exit 1
fi

# Check if firmware image exists
FIRMWARE="ultimate_mtd6.img"

if [ ! -f "$FIRMWARE" ]; then
    # Try in /tmp
    if [ -f "/tmp/$FIRMWARE" ]; then
        FIRMWARE="/tmp/$FIRMWARE"
    # Try on SD card
    elif [ -f "/mnt/sd/$FIRMWARE" ]; then
        FIRMWARE="/mnt/sd/$FIRMWARE"
    else
        print_error "Firmware-Image nicht gefunden: $FIRMWARE"
        echo ""
        echo "Bitte stelle sicher dass ${BOLD}ultimate_mtd6.img${NC} im gleichen"
        echo "Verzeichnis liegt oder kopiere es nach /tmp/ oder /mnt/sd/"
        echo ""
        echo "Download von: https://datenretter-pro.de/qianniao-firmware"
        exit 1
    fi
fi

print_success "Firmware-Image gefunden: $FIRMWARE"
SIZE=$(du -h "$FIRMWARE" | cut -f1)
print_success "Größe: $SIZE"

# Check MTD6
if [ ! -c "/dev/mtd6" ]; then
    print_error "MTD6 Partition nicht gefunden!"
    exit 1
fi

print_success "MTD6 Partition gefunden"
echo ""

# Warning
echo -e "${YELLOW}╔════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${YELLOW}║${NC}  ${BOLD}⚠️  WARNUNG  ⚠️${NC}                                                   ${YELLOW}║${NC}"
echo -e "${YELLOW}╚════════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo "Dieses Script wird:"
echo ""
echo "  1. Ein Backup von MTD6 erstellen"
echo "  2. Die aktuelle Firmware mit der neuen ersetzen"
echo "  3. Die Kamera neustarten"
echo ""
echo "Nach dem Flash:"
echo ""
echo "  ✓ ONVIF aktiviert (Port 6688)"
echo "  ✓ RTSP freigeschaltet (Port 8554)"
echo "  ✓ SSH/Telnet enabled"
echo "  ✓ Cloud geblockt"
echo "  ✓ Root-Zugang: root/root"
echo "  ✓ Admin: admin/admin123456"
echo ""
echo -e "${BOLD}Fortfahren? (ja/nein)${NC}"
read -r CONFIRM

if [ "$CONFIRM" != "ja" ]; then
    echo ""
    print_error "Abgebrochen"
    exit 0
fi

echo ""
print_step "Schritt 1/4: Backup erstellen"

BACKUP_DIR="/tmp/backup_$(date +%Y%m%d_%H%M%S)"
mkdir -p "$BACKUP_DIR"

dd if=/dev/mtd6 of="$BACKUP_DIR/mtd6_original.img" bs=1024 2>/dev/null

if [ $? -eq 0 ]; then
    print_success "Backup erstellt: $BACKUP_DIR/mtd6_original.img"
    
    # Try SD card backup
    if [ -d "/mnt/sd" ]; then
        cp -r "$BACKUP_DIR" /mnt/sd/ 2>/dev/null
        if [ $? -eq 0 ]; then
            print_success "Backup auch auf SD-Karte gesichert"
        fi
    fi
else
    print_error "Backup fehlgeschlagen!"
    exit 1
fi

echo ""
print_step "Schritt 2/4: MTD6 Partition löschen"

flash_erase /dev/mtd6 0 0 2>/dev/null

if [ $? -eq 0 ]; then
    print_success "Partition gelöscht"
else
    print_error "Löschen fehlgeschlagen!"
    exit 1
fi

echo ""
print_step "Schritt 3/4: Neue Firmware flashen"
echo "   (Dies dauert ca. 30 Sekunden...)"

dd if="$FIRMWARE" of=/dev/mtd6 bs=1024 2>/dev/null

if [ $? -eq 0 ]; then
    print_success "Firmware erfolgreich geflasht!"
    sync
else
    print_error "Flash fehlgeschlagen!"
    echo ""
    print_warning "Backup kann wiederhergestellt werden:"
    echo "  dd if=$BACKUP_DIR/mtd6_original.img of=/dev/mtd6"
    exit 1
fi

echo ""
print_step "Schritt 4/4: Zusammenfassung"

echo ""
echo -e "${GREEN}╔════════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║${NC}  ${BOLD}✓ FLASH ERFOLGREICH!${NC}                                            ${GREEN}║${NC}"
echo -e "${GREEN}╚════════════════════════════════════════════════════════════════════╝${NC}"
echo ""

# Get IP
IP=$(ifconfig eth0 2>/dev/null | grep 'inet addr' | cut -d: -f2 | awk '{print $1}')
if [ -z "$IP" ]; then
    IP=$(ip addr show eth0 2>/dev/null | grep 'inet ' | awk '{print $2}' | cut -d/ -f1)
fi

if [ -z "$IP" ]; then
    IP="<DEINE_KAMERA_IP>"
fi

echo -e "${BOLD}Nach dem Neustart verfügbar:${NC}"
echo ""
echo "  ${CYAN}ONVIF:${NC}"
echo "    http://$IP:6688/onvif/device_service"
echo ""
echo "  ${CYAN}RTSP:${NC}"
echo "    rtsp://admin:admin123456@$IP:8554/profile1"
echo ""
echo "  ${CYAN}SSH:${NC}"
echo "    ssh root@$IP"
echo "    Password: root"
echo ""
echo "  ${CYAN}Credentials:${NC}"
echo "    Root:  root/root"
echo "    Admin: admin/admin123456"
echo "    Super: admin/qianniaosuper"
echo ""
echo -e "${BOLD}Backup:${NC}"
echo "  $BACKUP_DIR/mtd6_original.img"

if [ -d "/mnt/sd/$(basename $BACKUP_DIR)" ]; then
    echo "  /mnt/sd/$(basename $BACKUP_DIR)/ (SD-Karte)"
fi

echo ""
echo -e "${BOLD}Wiederherstellung (falls nötig):${NC}"
echo "  dd if=$BACKUP_DIR/mtd6_original.img of=/dev/mtd6"
echo "  reboot"
echo ""

echo -e "${BOLD}Kamera jetzt neustarten? (ja/nein)${NC}"
read -r REBOOT

if [ "$REBOOT" = "ja" ]; then
    echo ""
    echo -e "${CYAN}Starte Kamera neu...${NC}"
    echo -e "${CYAN}Warte 1-2 Minuten, dann teste die Dienste!${NC}"
    echo ""
    sync
    sleep 2
    reboot
else
    echo ""
    echo -e "${CYAN}Neustart später mit:${NC} ${BOLD}reboot${NC}"
    echo ""
fi

echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
echo -e "  Tool von ${BOLD}datenretter-pro.de${NC} - Version $VERSION"
echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
echo ""
