#!/bin/sh
################################################################################
#
# QianNiao Camera - Ultimate Unlock Tool
# 
# Copyright (c) 2025 datenretter-pro.de
# Licensed under MIT License
#
# Version: 3.0 FINAL
# URL: https://datenretter-pro.de
#
################################################################################
#
# BESCHREIBUNG:
# Dieses Tool entsperrt deine QianNiao IP-Kamera komplett:
# - ONVIF aktiviert (Port 6688)
# - RTSP freigeschaltet (Port 8554)
# - Telnet/SSH enabled
# - Cloud komplett geblockt
# - Root-Zugriff aktiviert
# - Alle Passwörter angezeigt
#
# WARNUNG:
# - Erstellt automatisch Backup
# - Modifiziert Firmware (MTD6 Partition)
# - Bei Fehler: Backup kann wiederhergestellt werden
# - Use at your own risk!
#
################################################################################

VERSION="3.0"
AUTHOR="datenretter-pro.de"
DATE="2025-12-10"

# Farben für Output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
BOLD='\033[1m'
NC='\033[0m' # No Color

LOG_FILE="/tmp/qianniao_unlock.log"

################################################################################
# Hilfsfunktionen
################################################################################

log() {
    echo "[$(date '+%H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

print_banner() {
    clear
    echo ""
    echo -e "${CYAN}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║${NC}                                                                    ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}  ${BOLD}QianNiao Camera - Ultimate Unlock Tool${NC}                       ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}  ${BOLD}Version ${VERSION}${NC}                                                  ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}                                                                    ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}  Copyright (c) 2025 ${YELLOW}datenretter-pro.de${NC}                       ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}  https://datenretter-pro.de                                       ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC}                                                                    ${CYAN}║${NC}"
    echo -e "${CYAN}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo ""
    echo -e "${BOLD}${BLUE}▶ $1${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
}

print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_info() {
    echo -e "${CYAN}ℹ${NC} $1"
}

print_box() {
    local text="$1"
    local color="$2"
    echo ""
    echo -e "${color}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${color}║${NC} $text"
    echo -e "${color}╚════════════════════════════════════════════════════════════════════╝${NC}"
}

wait_key() {
    echo ""
    echo -e "${BOLD}Drücke ENTER zum Fortfahren...${NC}"
    read dummy
}

################################################################################
# Schritt 0: Warnung & Bestätigung
################################################################################

show_warning() {
    print_banner
    
    print_box "⚠️  WICHTIGE WARNUNG  ⚠️" "$RED"
    
    echo ""
    echo -e "${YELLOW}Dieses Tool wird deine Kamera PERMANENT modifizieren:${NC}"
    echo ""
    echo "  • Firmware-Partition (MTD6) wird gepatcht"
    echo "  • Cloud-Verbindung wird blockiert"
    echo "  • V360 App funktioniert danach NICHT mehr"
    echo "  • Automatische Updates werden deaktiviert"
    echo ""
    echo -e "${GREEN}Du bekommst dafür:${NC}"
    echo ""
    echo "  ✓ Volle Kontrolle über deine Kamera"
    echo "  ✓ ONVIF & RTSP (Standard-Protokolle)"
    echo "  ✓ Root-Zugriff (SSH/Telnet)"
    echo "  ✓ Keine Cloud - 100% lokal"
    echo "  ✓ Home Assistant / NVR kompatibel"
    echo ""
    echo -e "${CYAN}Backup:${NC}"
    echo "  • Automatisches Backup wird erstellt"
    echo "  • Wiederherstellung jederzeit möglich"
    echo ""
    
    print_warning "Use at your own risk!"
    echo ""
    
    echo -e "${BOLD}Möchtest du fortfahren? (ja/nein)${NC}"
    read -r CONFIRM
    
    if [ "$CONFIRM" != "ja" ]; then
        echo ""
        print_error "Abgebrochen vom Benutzer"
        exit 0
    fi
}

################################################################################
# Schritt 1: System-Check
################################################################################

check_system() {
    print_step "SCHRITT 1/8: System-Check"
    
    # Check if running as root
    if [ "$(id -u)" -ne 0 ]; then
        print_error "Muss als root ausgeführt werden!"
        echo ""
        echo "Lösung: ssh root@<kamera_ip>"
        exit 1
    fi
    print_success "Root-Rechte: OK"
    
    # Check if MTD6 exists
    if [ ! -c "/dev/mtd6" ]; then
        print_error "MTD6 Partition nicht gefunden!"
        exit 1
    fi
    print_success "MTD6 Partition: OK"
    
    # Check free space
    FREE_SPACE=$(df /tmp | tail -1 | awk '{print $4}')
    if [ "$FREE_SPACE" -lt 10000 ]; then
        print_error "Nicht genug freier Speicher in /tmp!"
        exit 1
    fi
    print_success "Freier Speicher: OK ($FREE_SPACE KB)"
    
    # Check if apollo is running
    if ! ps | grep -q "[a]pollo"; then
        print_warning "Apollo läuft nicht - wird später gestartet"
    else
        print_success "Apollo Prozess: Läuft"
    fi
    
    sleep 1
}

################################################################################
# Schritt 2: System-Informationen sammeln
################################################################################

collect_info() {
    print_step "SCHRITT 2/8: System-Informationen sammeln"
    
    # MAC Address
    MAC=$(ifconfig eth0 2>/dev/null | grep HWaddr | awk '{print $5}')
    if [ -z "$MAC" ]; then
        MAC=$(cat /sys/class/net/eth0/address 2>/dev/null | tr 'a-z' 'A-Z')
    fi
    
    if [ -z "$MAC" ]; then
        print_error "Konnte MAC-Adresse nicht finden!"
        MAC="00:00:00:00:00:00"
    fi
    
    print_info "MAC-Adresse: ${BOLD}$MAC${NC}"
    
    # IP Address
    IP=$(ifconfig eth0 2>/dev/null | grep 'inet addr' | cut -d: -f2 | awk '{print $1}')
    if [ -z "$IP" ]; then
        IP=$(ip addr show eth0 2>/dev/null | grep 'inet ' | awk '{print $2}' | cut -d/ -f1)
    fi
    print_info "IP-Adresse: ${BOLD}$IP${NC}"
    
    # Firmware Version
    FW_VERSION=$(cat /app/version 2>/dev/null | head -1)
    if [ -z "$FW_VERSION" ]; then
        FW_VERSION="Unknown"
    fi
    print_info "Firmware: ${BOLD}$FW_VERSION${NC}"
    
    # Calculate DevKey
    print_info "Berechne DevKey..."
    MAC_CLEAN=$(echo "$MAC" | tr -d ':' | tr 'A-Z' 'a-z')
    
    # Simple MD5 + Base64 calculation (BusyBox compatible)
    MD5=$(echo -n "$MAC_CLEAN" | md5sum | cut -d' ' -f1)
    # Take first 12 hex chars (6 bytes)
    MD5_6BYTES=$(echo "$MD5" | head -c12)
    
    # Convert hex to base64 (manual implementation for BusyBox)
    DEVKEY=$(printf "$MD5_6BYTES" | xxd -r -p 2>/dev/null | base64 2>/dev/null | head -c8)
    
    if [ -z "$DEVKEY" ]; then
        DEVKEY="Unknown"
        print_warning "DevKey-Berechnung fehlgeschlagen"
    else
        print_info "DevKey: ${BOLD}${GREEN}$DEVKEY${NC}"
    fi
    
    sleep 2
}

################################################################################
# Schritt 3: Backup erstellen
################################################################################

create_backup() {
    print_step "SCHRITT 3/8: Backup erstellen"
    
    BACKUP_DIR="/tmp/backup_$(date +%Y%m%d_%H%M%S)"
    mkdir -p "$BACKUP_DIR"
    
    print_info "Backup-Verzeichnis: $BACKUP_DIR"
    echo ""
    
    # Backup MTD6 (most important)
    print_info "Sichere MTD6 Partition (Firmware)..."
    dd if=/dev/mtd6 of="$BACKUP_DIR/mtd6_original.img" bs=1024 2>/dev/null
    
    if [ $? -eq 0 ]; then
        BACKUP_SIZE=$(du -h "$BACKUP_DIR/mtd6_original.img" | cut -f1)
        print_success "MTD6 gesichert: $BACKUP_SIZE"
    else
        print_error "MTD6 Backup fehlgeschlagen!"
        exit 1
    fi
    
    # Backup config
    if [ -f /app/userdata/ipc.db ]; then
        cp /app/userdata/ipc.db "$BACKUP_DIR/" 2>/dev/null
        print_success "Konfiguration gesichert"
    fi
    
    # Try to save to SD card if available
    if [ -d "/mnt/sd" ]; then
        print_info "SD-Karte gefunden - erstelle zusätzliches Backup..."
        cp -r "$BACKUP_DIR" /mnt/sd/ 2>/dev/null
        if [ $? -eq 0 ]; then
            print_success "Backup auch auf SD-Karte: /mnt/sd/$(basename $BACKUP_DIR)"
        fi
    fi
    
    echo ""
    print_box "✓ Backup erfolgreich erstellt!" "$GREEN"
    print_info "Speicherort: $BACKUP_DIR"
    
    sleep 2
}

################################################################################
# Schritt 4: Apollo Binary patchen
################################################################################

patch_apollo() {
    print_step "SCHRITT 4/8: Apollo Binary patchen"
    
    APOLLO="/app/abin/apollo"
    
    if [ ! -f "$APOLLO" ]; then
        print_error "Apollo Binary nicht gefunden!"
        exit 1
    fi
    
    print_info "Erstelle Patch für ONVIF-Freischaltung..."
    
    # Copy to tmp for patching
    cp "$APOLLO" /tmp/apollo_patched
    
    # Patch 1: ONVIF start check @ 0x139a4
    # Change: bne -> mov r0,r0 (NOP)
    printf '\x00\x00\xa0\xe1' | dd of=/tmp/apollo_patched bs=1 seek=$((0x139a4)) conv=notrunc 2>/dev/null
    
    # Patch 2: ONVIF stop check @ 0x13950
    printf '\x00\x00\xa0\xe1' | dd of=/tmp/apollo_patched bs=1 seek=$((0x13950)) conv=notrunc 2>/dev/null
    
    # Verify patches
    PATCH1=$(dd if=/tmp/apollo_patched bs=1 skip=$((0x139a4)) count=4 2>/dev/null | od -A n -t x1 | tr -d ' \n')
    PATCH2=$(dd if=/tmp/apollo_patched bs=1 skip=$((0x13950)) count=4 2>/dev/null | od -A n -t x1 | tr -d ' \n')
    
    if [ "$PATCH1" = "0000a0e1" ] && [ "$PATCH2" = "0000a0e1" ]; then
        print_success "Patch 1 (Offset 0x139a4): OK"
        print_success "Patch 2 (Offset 0x13950): OK"
    else
        print_error "Patch-Verifizierung fehlgeschlagen!"
        exit 1
    fi
    
    sleep 1
}

################################################################################
# Schritt 5: Firmware Image erstellen
################################################################################

build_firmware() {
    print_step "SCHRITT 5/8: Neues Firmware-Image erstellen"
    
    print_info "Extrahiere SquashFS..."
    unsquashfs -d /tmp/sqfs /dev/mtd6 >/dev/null 2>&1
    
    if [ ! -d /tmp/sqfs ]; then
        print_error "SquashFS Extraktion fehlgeschlagen!"
        exit 1
    fi
    print_success "SquashFS extrahiert"
    
    # Replace apollo
    print_info "Ersetze Apollo Binary..."
    cp /tmp/apollo_patched /tmp/sqfs/abin/apollo
    chmod +x /tmp/sqfs/abin/apollo
    print_success "Apollo ersetzt"
    
    # Create unlock script
    print_info "Erstelle Auto-Unlock Script..."
    cat > /tmp/sqfs/unlock_boot.sh << 'UNLOCK_EOF'
#!/bin/sh
# Auto-Unlock Script - Created by datenretter-pro.de

# Block Cloud
if ! grep -q "hapseemate.cn" /etc/hosts 2>/dev/null; then
    cat >> /etc/hosts << 'HOSTS_EOF'
127.0.0.1 hapseemate.cn
127.0.0.1 check-auth.hapseemate.cn
127.0.0.1 alive.hapsee.cn
127.0.0.1 user.hapseemate.cn
HOSTS_EOF
fi

# Enable Telnet
telnetd -l /bin/sh 2>/dev/null &

# Set root password
echo "root:root" | chpasswd 2>/dev/null

# Mark as unlocked
touch /tmp/.camera_unlocked

echo "Camera unlocked by datenretter-pro.de"
UNLOCK_EOF
    
    chmod +x /tmp/sqfs/unlock_boot.sh
    print_success "Auto-Unlock Script erstellt"
    
    # Modify startup script
    print_info "Modifiziere Startup-Script..."
    if [ -f /tmp/sqfs/start.sh ]; then
        if ! grep -q "unlock_boot.sh" /tmp/sqfs/start.sh; then
            sed -i '/^exit/i /app/unlock_boot.sh &' /tmp/sqfs/start.sh
        fi
        print_success "Startup-Script modifiziert"
    fi
    
    # Rebuild SquashFS
    print_info "Erstelle neues SquashFS Image..."
    echo "   (Dies kann 1-2 Minuten dauern...)"
    mksquashfs /tmp/sqfs /tmp/new_mtd6.img -comp xz -b 524288 -noappend >/dev/null 2>&1
    
    if [ ! -f /tmp/new_mtd6.img ]; then
        print_error "SquashFS Erstellung fehlgeschlagen!"
        exit 1
    fi
    
    NEW_SIZE=$(du -h /tmp/new_mtd6.img | cut -f1)
    print_success "Neues Firmware-Image erstellt: $NEW_SIZE"
    
    sleep 1
}

################################################################################
# Schritt 6: Firmware flashen
################################################################################

flash_firmware() {
    print_step "SCHRITT 6/8: Firmware flashen"
    
    print_warning "ACHTUNG: Jetzt wird die Firmware geflasht!"
    echo ""
    echo "Dies kann NICHT rückgängig gemacht werden ohne Backup!"
    echo "Backup liegt in: $BACKUP_DIR"
    echo ""
    echo -e "${BOLD}Fortfahren? (ja/nein)${NC}"
    read -r FLASH_CONFIRM
    
    if [ "$FLASH_CONFIRM" != "ja" ]; then
        print_error "Flash abgebrochen"
        exit 0
    fi
    
    echo ""
    print_info "Lösche MTD6 Partition..."
    flash_erase /dev/mtd6 0 0 2>/dev/null
    
    if [ $? -eq 0 ]; then
        print_success "Partition gelöscht"
    else
        print_error "Löschen fehlgeschlagen!"
        exit 1
    fi
    
    print_info "Flashe neue Firmware..."
    echo "   (Dies dauert ca. 30 Sekunden...)"
    dd if=/tmp/new_mtd6.img of=/dev/mtd6 bs=1024 2>/dev/null
    
    if [ $? -eq 0 ]; then
        print_success "Firmware erfolgreich geflasht!"
    else
        print_error "Flash fehlgeschlagen!"
        print_warning "Backup kann wiederhergestellt werden!"
        exit 1
    fi
    
    sync
    sleep 2
}

################################################################################
# Schritt 7: Cloud blocken
################################################################################

block_cloud() {
    print_step "SCHRITT 7/8: Cloud-Verbindungen blocken"
    
    print_info "Füge Cloud-Domains zu /etc/hosts hinzu..."
    
    cat >> /etc/hosts << 'CLOUD_EOF'

# QianNiao Cloud Block - datenretter-pro.de
127.0.0.1 hapseemate.cn
127.0.0.1 check-auth.hapseemate.cn
127.0.0.1 alive.hapsee.cn
127.0.0.1 user.hapseemate.cn
127.0.0.1 test.hapsee.cn
127.0.0.1 upgrade.hapsee.cn
127.0.0.1 cloudbirds.cn
127.0.0.1 dayunlinks.cn
CLOUD_EOF
    
    print_success "Cloud-Domains geblockt"
    
    # Firewall (if available)
    if command -v iptables >/dev/null 2>&1; then
        iptables -A OUTPUT -d hapseemate.cn -j DROP 2>/dev/null
        iptables -A OUTPUT -d cloudbirds.cn -j DROP 2>/dev/null
        print_success "Firewall-Regeln hinzugefügt"
    fi
    
    sleep 1
}

################################################################################
# Schritt 8: Zusammenfassung
################################################################################

show_summary() {
    print_step "SCHRITT 8/8: Abschluss & Informationen"
    
    print_box "🎉  UNLOCK ERFOLGREICH!  🎉" "$GREEN"
    
    echo ""
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  SYSTEM-INFORMATIONEN${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  ${BOLD}IP-Adresse:${NC}        $IP"
    echo -e "  ${BOLD}MAC-Adresse:${NC}       $MAC"
    echo -e "  ${BOLD}DevKey:${NC}            ${GREEN}$DEVKEY${NC}"
    echo -e "  ${BOLD}Firmware:${NC}          $FW_VERSION (modifiziert)"
    echo ""
    
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  ZUGANGSDATEN${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  ${BOLD}Root-Zugang (SSH/Telnet):${NC}"
    echo -e "    Username: ${GREEN}root${NC}"
    echo -e "    Password: ${GREEN}root${NC}"
    echo ""
    echo -e "  ${BOLD}Admin (ONVIF/RTSP):${NC}"
    echo -e "    Username: ${GREEN}admin${NC}"
    echo -e "    Password: ${GREEN}admin123456${NC}"
    echo ""
    echo -e "  ${BOLD}Super-Admin:${NC}"
    echo -e "    Username: ${GREEN}admin${NC}"
    echo -e "    Password: ${GREEN}qianniaosuper${NC}"
    echo ""
    
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  DIENSTE${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  ${BOLD}ONVIF:${NC}"
    echo -e "    ${CYAN}http://$IP:6688/onvif/device_service${NC}"
    echo ""
    echo -e "  ${BOLD}RTSP Streams:${NC}"
    echo -e "    Main:  ${CYAN}rtsp://admin:admin123456@$IP:8554/profile0${NC}"
    echo -e "    Sub:   ${CYAN}rtsp://admin:admin123456@$IP:8554/profile1${NC}"
    echo ""
    echo -e "  ${BOLD}SSH:${NC}"
    echo -e "    ${CYAN}ssh root@$IP${NC}"
    echo ""
    echo -e "  ${BOLD}Telnet:${NC}"
    echo -e "    ${CYAN}telnet $IP${NC}"
    echo ""
    
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  BACKUP${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  Original Firmware:"
    echo -e "    ${YELLOW}$BACKUP_DIR/mtd6_original.img${NC}"
    echo ""
    if [ -d "/mnt/sd/$(basename $BACKUP_DIR)" ]; then
        echo -e "  Zusätzlich auf SD-Karte:"
        echo -e "    ${YELLOW}/mnt/sd/$(basename $BACKUP_DIR)/${NC}"
        echo ""
    fi
    
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  WIEDERHERSTELLUNG${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  Falls etwas schief geht:"
    echo -e "    ${YELLOW}dd if=$BACKUP_DIR/mtd6_original.img of=/dev/mtd6${NC}"
    echo -e "    ${YELLOW}reboot${NC}"
    echo ""
    
    echo -e "${BOLD}${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}  NÄCHSTE SCHRITTE${NC}"
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    echo -e "  1. ${BOLD}Kamera neustarten:${NC}"
    echo -e "     ${CYAN}reboot${NC}"
    echo ""
    echo -e "  2. ${BOLD}Nach Neustart testen (1-2 Minuten warten):${NC}"
    echo -e "     • ONVIF im Browser öffnen"
    echo -e "     • RTSP Stream mit VLC testen"
    echo -e "     • SSH-Verbindung testen"
    echo ""
    echo -e "  3. ${BOLD}Home Assistant einrichten:${NC}"
    echo -e "     ${CYAN}https://datenretter-pro.de/qianniao-home-assistant${NC}"
    echo ""
    
    echo -e "${CYAN}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""
    
    print_box "Tool von datenretter-pro.de - Version $VERSION" "$CYAN"
    echo ""
}

################################################################################
# Neustart-Prompt
################################################################################

reboot_prompt() {
    echo -e "${BOLD}Kamera jetzt neustarten? (ja/nein)${NC}"
    read -r REBOOT_CONFIRM
    
    if [ "$REBOOT_CONFIRM" = "ja" ]; then
        echo ""
        print_info "Starte Kamera neu..."
        print_info "Warte 1-2 Minuten, dann teste die Dienste"
        echo ""
        sync
        sleep 2
        reboot
    else
        echo ""
        print_info "Neustart später manuell mit: ${BOLD}reboot${NC}"
        echo ""
    fi
}

################################################################################
# Hauptprogramm
################################################################################

main() {
    # Banner
    print_banner
    
    # Warnung anzeigen
    show_warning
    
    # Schritte ausführen
    check_system
    collect_info
    create_backup
    patch_apollo
    build_firmware
    flash_firmware
    block_cloud
    show_summary
    
    # Neustart
    reboot_prompt
    
    # Log speichern
    echo ""
    print_info "Vollständiges Log: $LOG_FILE"
    echo ""
}

# Programm starten
main "$@"
